using System;
using System.Data;

using gov.va.med.vbecs.DAL.VistALink.OpenLibrary;

namespace gov.va.med.vbecs.DAL.VAL
{	
	#region Header

	///<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
	///<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
	///<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
	///<Developers>
	///	<Developer>Stas Antropov</Developer>
	///</Developers>
	///<SiteName>Hines OIFO</SiteName>
	///<CreationDate>2/27/2005</CreationDate>
	///<Note>The Food and Drug Administration classifies this software as a medical device.  As such, it may not be changed in any way. Modifications to this software may result in an adulterated medical device under 21CFR820, the use of which is considered to be a violation of US Federal Statutes.  Acquiring and implementing this software through the Freedom of information Act requires the implementor to assume total responsibility for the software, and become a registered manufacturer of a medical device, subject to FDA regulations</Note>
	///<summary>
	///		Provides means for getting patient medication profile 
	///		information from VistA system via VistALink.
	///	</summary>

	#endregion

	public class MedicationProfile
	{
		private const string MEDICATION_PROFILE_LOOKUP_RPC_NAME = "VBECS MED PROFILE LOOKUP";
		private const string MEDICATION_PROFILE_TABLENAME = "MedicationProfile";
		private const string MEDICATION_PROFILE_ORDER_TABLENAME = "Order";

		private MedicationProfile() {}

		///<Developers>
		///	<Developer>Brian Tomlin</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/26/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4532"> 
		///		<ExpectedInput>'266763' as vistaPatientId, a DateTime of 3 years past for startDate,
		///		Today for the endDate.</ExpectedInput>
		///		<ExpectedOutput>Non-empty DataSet with medications from VistA, or the value of
		///		'No medications found' in the Name column of the Order table in the DataSet.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="4536"> 
		///		<ExpectedInput>'266763' as vistaPatientId, a DateTime.MinValue for startDate,
		///		DateTime.MinValue for the endDate.</ExpectedInput>
		///		<ExpectedOutput>Non-empty DataSet with medications from VistA, or the value of
		///		'No medications found' in the Name column of the Order table in the DataSet.
		///		BeginningSearchDate of the MedicationProfile table should be set to 180 days 
		///		prior and EndingSearchDate should be set to the current date.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="4537"> 
		///		<ExpectedInput>'999999999999999' as vistaPatientId, valid DateTime for startDate,
		///		valid DateTime for endDate.</ExpectedInput>
		///		<ExpectedOutput>The Patient column of the MedicationProfile table returned should
		///		contain the text 'Invalid Patient ID'.  No other columns in any table should contain
		///		data.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4533"> 
		///		<ExpectedInput>Int.MinValue as the vistaPatientId, valid DateTime values for startDate
		///		and EndDate.</ExpectedInput>
		///		<ExpectedOutput>ArgumentOutOfRangeException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Queries VistA system executing the VBECS MED PROFILE LOOKUP Remote Procedure and returns
		/// a read-only <see cref="DataSet"/> containing search results. Search is performed by supplied parameters.
		/// </summary>
		/// <param name="vistaPatientId">(Required) The VistA Patient ID (DUZ) from the Patient Table in VBECS</param>
		/// <param name="startDate">(Optional) The date used to begin the search for patient medications. Defaults to 180 days in the past.</param>
		/// <param name="endDate">(Optional) The date used to end the search for patient medications. Defaults to current date.</param>
		/// <returns></returns>
		/// CR 2431 changed vistaPatientId from int to long.
		public static DataSet GetMedicationProfile(long vistaPatientId, DateTime startDate, DateTime endDate)
		{
			if( vistaPatientId < 0 )
				throw( new ArgumentOutOfRangeException( "vistaPatientId" ) );

			if( startDate > DateTime.Now || startDate == DateTime.MinValue )
			{
				startDate = DateTime.Now.Subtract(System.TimeSpan.FromDays( 180 ) );
			}

			if( endDate < startDate || endDate > DateTime.Now )
			{
				endDate = DateTime.Now;
			}

			return GetMedicationProfileData( vistaPatientId.ToString(),
				Common.HL7DateFormat.ConvertDateTimeToHL7( startDate ),
				Common.HL7DateFormat.ConvertDateTimeToHL7( endDate ) );
		}

		/// <summary>
		/// Executes Medication Profile Lookup RPC with given parameters
		/// and gets results in a read-only <see cref="DataSet"/>.
		/// </summary>
		/// <param name="vistaPatientId">VistA Patient ID (DUZ) from the Patient Table in VBECS</param>
		/// <param name="startDate">Date string in HL7 format returned from <see cref="Common.HL7DateFormat.ConvertDateTimeToHL7"/>
		/// used to perform lookup.</param>
		/// <param name="endDate">Date string in HL7 format returned from <see cref="Common.HL7DateFormat.ConvertDateTimeToHL7"/>
		/// used to perform lookup.</param>
		/// <returns></returns>
		private static DataSet GetMedicationProfileData( string vistaPatientId, string startDate, string endDate )
		{
			DataSet _ds = new DataSet();
			DataSet _dsReturn = new DataSet();
			DataTable _dtMedProf = GetMedicationProfileDataTable();
			DataTable _dtOrder = GetOrderDataTable();

			string _rpcResult = VistALink.GetRpcBroker().ExecuteRpc( GetMedicationProfileRpcRequest( vistaPatientId, startDate, endDate ) );

			_ds = DataTransformUtility.LoadXmlStringIntoDataSet( _rpcResult );

			foreach( DataTable dt in _ds.Tables )
			{
				if( dt.TableName == MEDICATION_PROFILE_TABLENAME )
				{
					foreach( DataRow dr in dt.Rows )
					{
						DataRow _drMedProf = _dtMedProf.NewRow();
						_drMedProf["PatientName"] = System.DBNull.Value;
						_drMedProf["PatientSsn"] = System.DBNull.Value;
						_drMedProf["PatientDob"] = System.DBNull.Value;
						_drMedProf["BeginningSearchDate"] = System.DBNull.Value;
						_drMedProf["EndingSearchDate"] = System.DBNull.Value;
						_dtMedProf.Rows.Add(_drMedProf);

						if( dr.Table.Columns.Contains("PatientName") && dr["PatientName"].ToString() != string.Empty )
						{
							_drMedProf["PatientName"] = dr["PatientName"].ToString();
						}
						if( dr.Table.Columns.Contains("PatientSsn") && dr["PatientSsn"].ToString() != string.Empty )
						{
							_drMedProf["PatientSsn"] = dr["PatientSsn"].ToString();
						}
						if( dr.Table.Columns.Contains("PatientDob") && dr["PatientDob"].ToString() != string.Empty )
						{
							_drMedProf["PatientDob"] = Common.VBECSDateTime.FormatDateString(Common.HL7DateFormat.ConvertHL7DateTime(dr["PatientDob"].ToString()));
						}
						if( dr.Table.Columns.Contains("BeginningSearchDate") && dr["BeginningSearchDate"].ToString() != string.Empty )
						{
							_drMedProf["BeginningSearchDate"] = Common.VBECSDateTime.FormatDateString(Common.HL7DateFormat.ConvertHL7DateTime(dr["BeginningSearchDate"].ToString()));
						}
						if( dr.Table.Columns.Contains("EndingSearchDate") && dr["EndingSearchDate"].ToString() != string.Empty )
						{
							_drMedProf["EndingSearchDate"] = Common.VBECSDateTime.FormatDateString(Common.HL7DateFormat.ConvertHL7DateTime(dr["EndingSearchDate"].ToString()));
						}
					}
				}

				if( dt.TableName == MEDICATION_PROFILE_ORDER_TABLENAME )
				{
					foreach( DataRow dr in dt.Rows )
					{
						DataRow _drOrder = _dtOrder.NewRow();
						_drOrder["OrderNumber"] = System.DBNull.Value;
						_drOrder["DrugName"] = System.DBNull.Value;
						_drOrder["IssueStartDate"] = System.DBNull.Value;
						_drOrder["Status"] = System.DBNull.Value;
						_drOrder["InpatientOutpatientIndicator"] = System.DBNull.Value;
						_dtOrder.Rows.Add(_drOrder);

						if( dr.Table.Columns.Contains("OrderNumber") && dr["OrderNumber"].ToString() != string.Empty )
						{
							_drOrder["OrderNumber"] = dr["OrderNumber"].ToString();
						}
						if( dr.Table.Columns.Contains("DrugName") && dr["DrugName"].ToString() != string.Empty )
						{
							_drOrder["DrugName"] = dr["DrugName"].ToString();
						}
						if( dr.Table.Columns.Contains("IssueStartDate") && dr["IssueStartDate"].ToString() != string.Empty )
						{
							_drOrder["IssueStartDate"] = Common.VBECSDateTime.FormatDateString(Common.HL7DateFormat.ConvertHL7DateTime(dr["IssueStartDate"].ToString()));
						}
						if( dr.Table.Columns.Contains("Status") && dr["Status"].ToString() != string.Empty )
						{
							_drOrder["Status"] = dr["Status"].ToString();
						}
						if( dr.Table.Columns.Contains("InpatientOutpatientIndicator") && dr["InpatientOutpatientIndicator"].ToString() != string.Empty )
						{
							_drOrder["InpatientOutpatientIndicator"] = dr["InpatientOutpatientIndicator"].ToString();
						}

					}
				}
			}
			_dsReturn.Tables.Add(_dtMedProf);
			_dsReturn.Tables.Add(_dtOrder);

			return _dsReturn;
		}

		/// <summary>
		/// Creates medication profile lookup RPC request from given search parameters.
		/// </summary>
		/// <param name="vistaPatientId">VistA Patient ID (DUZ) from the Patient Table in VBECS</param>
		/// <param name="startDate">Date string in HL7 format returned from <see cref="Common.HL7DateFormat.ConvertDateTimeToHL7"/>
		/// used to perform lookup.</param>
		/// <param name="endDate">Date string in HL7 format returned from <see cref="Common.HL7DateFormat.ConvertDateTimeToHL7"/>
		/// used to perform lookup.</param>
		/// <returns>Medication Profile RpcRequest object</returns>
		private static RpcRequest GetMedicationProfileRpcRequest( string vistaPatientId, string startDate, string endDate )
		{
			RpcRequest _medProfileLookupRequest = VistALink.CreateRpcRequest( MEDICATION_PROFILE_LOOKUP_RPC_NAME );
			_medProfileLookupRequest.Parameters.Add( new RpcStringParameter( 1, vistaPatientId ) );
			_medProfileLookupRequest.Parameters.Add( new RpcStringParameter( 2, startDate ) );
			_medProfileLookupRequest.Parameters.Add( new RpcStringParameter( 3, endDate ) );

			return _medProfileLookupRequest;
		}

		private static DataTable GetMedicationProfileDataTable()
		{
			DataTable _dt = new DataTable();

			// MedicationProfile DataTable
			_dt.TableName = "MedicationProfile";
			_dt.Columns.Add("PatientName");
			_dt.Columns.Add("PatientSsn");
			_dt.Columns.Add("PatientDob");
			_dt.Columns.Add("BeginningSearchDate");
			_dt.Columns.Add("EndingSearchDate");

			return _dt;
		}

		private static DataTable GetOrderDataTable()
		{
			DataTable _dt = new DataTable();

			// Order DataTable
			_dt.TableName = "Order";
			_dt.Columns.Add("OrderNumber");
			_dt.Columns.Add("DrugName");
			_dt.Columns.Add("IssueStartDate");
			_dt.Columns.Add("Status");
			_dt.Columns.Add("InpatientOutpatientIndicator");

			return _dt;
		}
	}
}
